<?php

namespace App\Http\Controllers;

use App\Models\Auditionee;
use Illuminate\Support\Facades\Schema;
use Illuminate\Http\Request;
use App\Enums\Country;

class ExportController extends Controller
{
    public function exportCsv()
    {
        // Define the CSV file name
        $fileName = 'auditionees_export.csv';

        // Fetch all auditionees with related castings and pastoral data
        $auditionees = Auditionee::with(['casting', 'pastoral'])->get();

        // Fetch columns from the tables (adjust table names if needed)
        $auditioneesColumns = Schema::getColumnListing('auditionees');
        $auditioneeCastingsColumns = Schema::getColumnListing('auditionee_castings');
        $auditioneePastoralColumns = Schema::getColumnListing('auditionee_pastorals');

        // Create headers for the CSV by combining column names
        $csvHeaders = array_merge(
            $auditioneesColumns,
            $auditioneeCastingsColumns,
            $auditioneePastoralColumns
        );

        // Create headers for the CSV file
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="'.$fileName.'"',
        ];

        // Create callback function to handle the CSV output
        $callback = function () use ($auditionees, $csvHeaders, $auditioneesColumns, $auditioneeCastingsColumns, $auditioneePastoralColumns) {
            $file = fopen('php://output', 'w');

            // Write the CSV column headers
            fputcsv($file, $csvHeaders);

            // Loop through auditionees and their related data
            foreach ($auditionees as $auditionee) {
                // Get auditionee attributes
                $auditioneeData = [];
                foreach ($auditioneesColumns as $column) {
                    $value = $auditionee->$column;
                    
                    if ($column == "country") {
                        $auditioneeData[] = Country::from($value->value)->name;
                    } else {
                        $auditioneeData[] = $value;
                    }
                }

                // Initialize casting and pastoral data arrays
                $castingData = [];
                $pastoralData = [];

                // Check if casting data exists and add it
                if ($auditionee->casting) {
                    foreach ($auditioneeCastingsColumns as $column) {
                        $value = $auditionee->casting->$column;

                        // Handle enums in casting
                        if (is_object($value) && method_exists($value, 'value')) {
                            $castingData[] = $value->value;
                        } else {
                            $castingData[] = $value;
                        }
                    }
                } else {
                    // If casting doesn't exist, fill with empty values
                    $castingData = array_fill(0, count($auditioneeCastingsColumns), '');
                }

                // Check if pastoral data exists and add it
                if ($auditionee->pastoral) {
                    foreach ($auditioneePastoralColumns as $column) {
                        $value = $auditionee->pastoral->$column;

                        // Handle enums in pastoral
                        if (is_object($value) && method_exists($value, 'value')) {
                            $pastoralData[] = $value->value;
                        } else {
                            $pastoralData[] = $value;
                        }
                    }
                } else {
                    // If pastoral doesn't exist, fill with empty values
                    $pastoralData = array_fill(0, count($auditioneePastoralColumns), '');
                }

                // Combine auditionee, casting, and pastoral data into a single row
                $row = array_merge($auditioneeData, $castingData, $pastoralData);

                // Write the row to the CSV
                fputcsv($file, $row);
            }

            fclose($file);
        };

        // Return the CSV file as a response with the appropriate headers
        return response()->stream($callback, 200, $headers);
    }
}